'use strict';

var fs = require('fs');
var path = require('path');
var os = require('os');
var net = require('net');

const debug = require('debug')('@sap/hana-client:index');
debug.log = console.info.bind(console);

const name = 'index.js';

debug('Starting %s', name);

var extensions = {
    'darwin': '.dylib',
    'linux': '.so',
    'win32': '.dll'
};

var db = null;

function isMusl() {
    const output = require('child_process').spawnSync('ldd',['--version']).stderr.toString('utf8');
    if(output.indexOf('musl') > -1) {
        return true;
    }
    return false;
}

var readPipe = null;

function __refPipe() {
    readPipe.ref();
}

function __unrefPipe() {
    readPipe.unref();
}

function setupPipe(pipeInfo) {
    if (!db) return;
    var ldb = db;
    if (process.platform === 'win32') {
        readPipe = net.createConnection(pipeInfo["IPCEndpoint"]);
    } else {
        readPipe = net.Socket({fd       : pipeInfo["fd"],
                               readable : true});
    }
    readPipe.unref();
    readPipe.on('data', function(data) {
        ldb.__runCallbacks();
    });
    ldb.__setRefFunctions(__refPipe, __unrefPipe);
}

process.on('exit', function() {
    db.__stopAllWork();
});

debug('Detected os.arch=' + os.arch() + ', '
        + 'process.platform=' + process.platform + ', '
        + 'process.arch=' + process.arch + ', '
        + 'process.version=' + process.version );

if (process.platform === 'win32' && process.arch != 'x64') {
    debug('ERROR: On Windows, this driver only supports the x64 architecture. ' +
        'Your node process is: ' + process.arch );
    throw new Error('On Windows, this driver only supports the x64 architecture. ' +
        'Your node process is: ' + process.arch );
}

var majorVersion = process.versions.node.split('.')[0];
if (majorVersion < 8) {
    throw new Error("Node version " + process.version + " is unsupported by @sap/hana-client. Only versions >=8 are supported");
}

// Look for prebuilt binary and DBCAPI based on platform
var pb_subdir = null;
var dbcapi_name = 'libdbcapiHDB';
if (process.platform === 'linux') {
    if ( isMusl() ) {
        pb_subdir = 'linuxx86_64_alpine-gcc6';
    } else if (process.arch === 'x64') {
        pb_subdir = 'linuxx86_64-gcc48';
    } else if (process.arch.toLowerCase().indexOf('ppc') != -1 && os.endianness() === 'LE') {
        pb_subdir = 'linuxppc64le-gcc48';
    } else {
        pb_subdir = 'linuxppc64-gcc48';
    }

} else if (process.platform === 'win32') {
    pb_subdir = 'ntamd64-msvc2010';
} else if (process.platform === 'darwin') {
    pb_subdir = 'darwinintel64-xcode7';
}
var dbcapi_env = 'DBCAPI_API_DLL';

var modpath = path.dirname(__dirname);
var pb_path = path.join(modpath, 'prebuilt', pb_subdir);
var dbcapi = path.join(pb_path, dbcapi_name + extensions[process.platform]);
try {
    fs.statSync(dbcapi);
} catch (ex) {
    // No binary!
    debug(ex.message);
    debug("No DBCAPI interface driver found for platform: '" +
        process.platform + "', arch: '" + process.arch +
        "', endianness: '" + os.endianness() + "' for Node version: '" +
        process.version + "'");
    debug("You need to set the " + dbcapi_env + " environment variable to point to " +
        dbcapi_name + extensions[process.platform] + " for this platform.");
    throw new Error('`' + dbcapi + '` is missing.');
}

// Remove unrelated native platform libraries
const removeDirectory = function (dir) {
    if (fs.existsSync(dir)) {
        fs.readdirSync(dir).forEach((file, index) => {
            const curPath = path.join(dir, file);
            if (fs.lstatSync(curPath).isDirectory()) { // recurse
                removeDirectory(curPath);
            } else { // delete file
                fs.unlinkSync(curPath);
            }
        });
        fs.rmdirSync(dir);
    }
};
const getSubDirs = function (dir) {
    const isDirectory = source => fs.lstatSync(source).isDirectory();
    const getDirectories = source =>
        fs.readdirSync(source).map(name => path.join(source, name)).filter(isDirectory);
    return getDirectories(dir);
};
if (process.env['HDB_NODE_PLATFORM_CLEAN'] === '1') {
    var pb_path_all = getSubDirs(path.join(modpath, 'prebuilt'));
    pb_path_all.forEach((pbPath) => {
        if (pbPath !== pb_path) {
            try {
                removeDirectory(pbPath);
            } catch (ex) {
                debug(ex.message);
                debug("Failed to remove unrelated native platform libraries in '" + pbPath + "'.");
            }
        }
    });
}

// Found dbcapi
process.env[dbcapi_env] = dbcapi;

// Now find driver
var default_driver_file = 'hana-client';
var driver_file = default_driver_file;

if (process.env['HDB_NODE_NO_NAPI_DRIVER'] === '1') {
    // Check if there is a node-version-specific driver
    // Fall back on hana-client.node
    var v = process.version;
    var match = v.match(/v([0-9]+)\.([0-9]+)\.[0-9]+/);
    driver_file += '_v' + match[1];
    if (match[1] + 0 == 0) {
        driver_file += '_' + match[2];
    }
}
var driver_path = path.join(pb_path, driver_file + '.node');

try {
    debug("Checking for existence of "+driver_path);
    fs.statSync(driver_path);
} catch (ex) {
    debug("Did not find "+driver_path);
    driver_path = path.join(pb_path, default_driver_file + '.node');
    try {
        debug("Checking for existence of "+driver_path);
        fs.statSync(driver_path);
    } catch (ex) {
        debug("No prebuilt node driver found for platform: '" +
            process.platform + "', arch: '" + process.arch +
            "', endianness: '" + os.endianness() + "' for Node version: '" +
            process.version + "'");
    }
}

// Try loading
// 1. User's build
// 2. Prebuilt

debug('Attempting to load Hana node-hdbcapi driver');

var userbld_driver_path = path.join(modpath, 'build', 'Release', 'hana-client.node');
debug('... Trying user-built copy...');
try {
    debug('... Looking for user-built copy in ' + userbld_driver_path + ' ... ');
    fs.statSync(userbld_driver_path);
    debug('Found.');
    try {
        debug('... Attempting to load user-built copy... ');
        db = require(userbld_driver_path);
        debug('Loaded.');
    } catch (ex) {
        debug(ex.message);
        debug('Could not load: User-built copy did not satisfy requirements.');
        throw ex;
    }
} catch (ex) {
    debug('Not found.');
}

if (db === null) {
    debug('... Trying prebuilt copy...');
    try {
        debug('... Looking for prebuilt copy in ' + driver_path + ' ... ');
        db = require(driver_path);
        debug('Loaded.');
    } catch (ex) {
        debug(ex.message);
        debug('Could not load: Prebuilt copy did not satisfy requirements.');
        debug("Could not load modules for Platform: '" +
            process.platform + "', Process Arch: '" + process.arch +
            "', and Version: '" + process.version + "'");
        throw ex;
    }
}

db.__loadDbcapi(dbcapi);
var pipeInfo = db.__getPipe();
setupPipe(pipeInfo);
if (db !== null) {
    debug('Success.');
}
module.exports = db;
